package com.foobar.dragbug;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;

import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.WorkbenchException;
import org.eclipse.ui.XMLMemento;
import org.eclipse.ui.internal.IWorkbenchConstants;
import org.eclipse.ui.internal.Workbench;
import org.eclipse.ui.internal.WorkbenchWindow;

public final class WorkbenchMemory {

	private static final String PREF_WORKBENCH_MEMENTO = "workbenchMemento"; //$NON-NLS-1$

	/**
	 * Stores the workbench state
	 */

	public static void memorize() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		IWorkbenchWindow window = workbench.getActiveWorkbenchWindow();
		if (window == null) {
			return; // something went very wrong in the application start
		}
		IPreferenceStore preferences = getUserPreferences();
		memorizeWorkbench(workbench, preferences);
	}

	private static void memorizeWorkbench(IWorkbench workbench, IPreferenceStore preferences) {
		XMLMemento memento = createMemento(workbench);
		saveMemento(memento, preferences);
	}

	@SuppressWarnings("restriction")
	private static XMLMemento createMemento(IWorkbench workbench) {
		// extracted from Workbench.saveState()
		XMLMemento memento = XMLMemento.createWriteRoot(IWorkbenchConstants.TAG_WORKBENCH);

		// Save the advisor state.
		IMemento advisorState = memento.createChild(IWorkbenchConstants.TAG_WORKBENCH_ADVISOR);
		Application.getWorkbenchAdvisor().saveState(advisorState);

		// Save the workbench windows.
		IWorkbenchWindow[] windows = workbench.getWorkbenchWindows();
		for (int nX = 0; nX < windows.length; nX++) {
			WorkbenchWindow window = (WorkbenchWindow) windows[nX];
			IMemento childMem = memento.createChild(IWorkbenchConstants.TAG_WINDOW);
			window.saveState(childMem);
		}
		((Workbench) workbench).getEditorHistory().saveState(memento.createChild(IWorkbenchConstants.TAG_MRU_LIST));

		return memento;
	}

	private static void saveMemento(XMLMemento memento, IPreferenceStore preferences) {
		String mementoString = ""; //$NON-NLS-1$
		try (StringWriter writer = new StringWriter();) {
			memento.save(writer);
			mementoString = writer.getBuffer().toString();
		} catch (IOException e) {
			// ignore errors
			e.printStackTrace();
		}
		preferences.setValue(PREF_WORKBENCH_MEMENTO, mementoString);
	}

	/**
	 * Revives the workbench state
	 */

	public static void remember() {
		IPreferenceStore preferences = getUserPreferences();
		rememberWorkbench(preferences);
	}

	private static void rememberWorkbench(IPreferenceStore preferences) {
		XMLMemento memento = loadMemento(preferences);
		if (memento != null) {
			restoreMemento(PlatformUI.getWorkbench(), memento);
		}
	}

	private static XMLMemento loadMemento(IPreferenceStore preferences) {
		String mementoString = preferences.getString(PREF_WORKBENCH_MEMENTO);
		if (mementoString != null && !mementoString.isEmpty()) {
			try (StringReader reader = new StringReader(mementoString)) {
				return XMLMemento.createReadRoot(reader);
			} catch (WorkbenchException e) {
				// ignore errors
				e.printStackTrace();
			}
		}
		return null;
	}

	@SuppressWarnings("restriction")
	private static void restoreMemento(IWorkbench workbench, XMLMemento memento) {
		// extracted from Workbench.restoreState()

		IMemento mruMemento = memento.getChild(IWorkbenchConstants.TAG_MRU_LIST);
		if (mruMemento != null) {
			((Workbench) workbench).getEditorHistory().restoreState(mruMemento);
		}
		// Restore advisor state.
		IMemento advisorState = memento.getChild(IWorkbenchConstants.TAG_WORKBENCH_ADVISOR);
		if (advisorState != null) {
			Application.getWorkbenchAdvisor().restoreState(advisorState);
		}

		// Get the child windows.
		IMemento[] children = memento.getChildren(IWorkbenchConstants.TAG_WINDOW);

		IWorkbenchWindow[] windows = workbench.getWorkbenchWindows();

		for (int nX = 0; nX < windows.length; nX++) {
			WorkbenchWindow window = (WorkbenchWindow) windows[nX];
			window.restoreState(children[nX], null);
		}
	}

	public static IPreferenceStore getUserPreferences() {
		return Activator.getDefault().getPreferenceStore();
	}

	private WorkbenchMemory() {
		// Just two words: Memory palace!
	}
}
